/**
 * Global registry for PostgreSQL clients.
 *
 * This module provides a way to track all active PostgreSQL clients
 * so they can be gracefully shut down together (e.g., on process exit).
 *
 * The runtime can use `shutdownAll()` to close all registered clients
 * during graceful shutdown.
 *
 * When @agentuity/runtime is available, this module automatically registers
 * a shutdown hook so all postgres clients are closed during graceful shutdown.
 */
import type { PostgresClient } from './client';
/**
 * Registers a client in the global registry.
 * Called automatically when a client is created.
 *
 * @param client - The client to register
 * @internal
 */
export declare function registerClient(client: PostgresClient): void;
/**
 * Unregisters a client from the global registry.
 * Called automatically when a client is closed.
 *
 * @param client - The client to unregister
 * @internal
 */
export declare function unregisterClient(client: PostgresClient): void;
/**
 * Returns the number of registered clients.
 * Useful for debugging and testing.
 */
export declare function getClientCount(): number;
/**
 * Returns all registered clients.
 * Useful for debugging and monitoring.
 */
export declare function getClients(): ReadonlySet<PostgresClient>;
/**
 * Shuts down all registered PostgreSQL clients gracefully.
 *
 * This function:
 * 1. Signals shutdown to all clients (prevents reconnection)
 * 2. Closes all clients in parallel
 * 3. Clears the registry
 *
 * This is intended to be called by the runtime during graceful shutdown.
 *
 * @param timeoutMs - Optional timeout in milliseconds. If provided, the function
 *                    will resolve after the timeout even if some clients haven't
 *                    finished closing. Default: no timeout.
 * @returns A promise that resolves when all clients are closed (or timeout)
 *
 * @example
 * ```typescript
 * import { shutdownAll } from '@agentuity/postgres';
 *
 * process.on('SIGTERM', async () => {
 *   await shutdownAll(5000); // 5 second timeout
 *   process.exit(0);
 * });
 * ```
 */
export declare function shutdownAll(timeoutMs?: number): Promise<void>;
/**
 * Checks if there are any active (non-shutdown) clients.
 * Useful for health checks.
 */
export declare function hasActiveClients(): boolean;
//# sourceMappingURL=registry.d.ts.map