import type { PaneAction, WindowState, TmuxConfig } from './types';
type OwnershipContext = {
    instanceId: string;
    ownerPid: number;
    serverKey: string;
    tmuxSessionId?: string;
};
type OwnershipLookup = {
    serverKey: string;
    instanceId: string;
    tmuxSessionId?: string;
};
export interface ActionResult {
    success: boolean;
    paneId?: string;
    windowId?: string;
    pid?: number;
    error?: string;
}
export declare function getPanePid(paneId: string): Promise<number | undefined>;
export declare function getPanePidSync(paneId: string): number | undefined;
export declare function findOwnedAgentPanes(serverKey: string): Promise<Array<{
    paneId: string;
    panePid?: number;
    sessionId?: string;
    instanceId?: string;
}>>;
/**
 * Kill a process and all its children (the entire process tree).
 *
 * This is necessary because we spawn `bash -c "opencode attach ...; tmux kill-pane"`
 * and #{pane_pid} returns the bash PID, not the opencode attach PID.
 * We need to kill the children (opencode attach) not just the parent (bash).
 */
export declare function killProcessByPid(pid: number): Promise<boolean>;
/**
 * Execute a single pane action
 *
 * All agents spawn in a dedicated "Agents" window with tiled grid layout.
 */
export declare function executeAction(action: PaneAction, ctx: {
    config: TmuxConfig;
    serverUrl: string;
    windowState: WindowState;
    ownership: OwnershipContext;
}): Promise<ActionResult>;
/**
 * Execute multiple actions in sequence
 */
export declare function executeActions(actions: PaneAction[], ctx: {
    config: TmuxConfig;
    serverUrl: string;
    windowState: WindowState;
    ownership: OwnershipContext;
}): Promise<{
    success: boolean;
    spawnedPaneId?: string;
    results: Array<{
        action: PaneAction;
        result: ActionResult;
    }>;
}>;
/**
 * Close a pane by its ID
 * Exported for use by TmuxSessionManager when sessions complete
 */
export declare function closePaneById(paneId: string, pid?: number): Promise<ActionResult>;
/**
 * Reset the agents window state (for cleanup)
 */
export declare function resetAgentsWindow(ownership?: OwnershipContext): void;
/**
 * Close the agents window if it exists
 * This kills the entire window, which closes all panes within it
 *
 * SAFETY: Verifies the window is named "Agents" before killing to prevent
 * accidentally killing user windows if the cached ID is stale.
 */
export declare function closeAgentsWindow(ownership?: OwnershipLookup): Promise<void>;
/**
 * Synchronously close the agents window (for shutdown)
 * Uses runTmuxCommandSync to ensure it completes before process exit
 *
 * SAFETY: Verifies the window is named "Agents" before killing to prevent
 * accidentally killing user windows if the cached ID is stale.
 */
export declare function closeAgentsWindowSync(ownership?: OwnershipLookup): void;
/**
 * Get the current agents window ID (for testing/debugging)
 */
export declare function getAgentsWindowId(ownership?: OwnershipContext): string | undefined;
/**
 * Clean up owned tmux windows/panes using ownership tags.
 */
export declare function cleanupOwnedResources(serverKey: string, instanceId: string): Promise<{
    panesClosed: number;
    windowClosed: boolean;
}>;
/**
 * Synchronous cleanup for owned tmux windows/panes.
 */
export declare function cleanupOwnedResourcesSync(serverKey: string, instanceId: string): {
    panesClosed: number;
    windowClosed: boolean;
};
export {};
//# sourceMappingURL=executor.d.ts.map