/**
 * Create a WebSocket client wrapper with event-based API.
 */
export function createWebSocketClient(url, protocols) {
    const ws = new WebSocket(url, protocols);
    const handlers = {
        message: new Set(),
        open: new Set(),
        close: new Set(),
        error: new Set(),
    };
    // Set up native WebSocket event listeners
    ws.addEventListener('message', (event) => {
        let data = event.data;
        // Try to parse JSON if data is a string
        if (typeof event.data === 'string') {
            try {
                data = JSON.parse(event.data);
            }
            catch {
                // Keep as string if not valid JSON
                data = event.data;
            }
        }
        handlers.message.forEach((handler) => handler(data));
    });
    ws.addEventListener('open', (event) => {
        handlers.open.forEach((handler) => handler(event));
    });
    ws.addEventListener('close', (event) => {
        handlers.close.forEach((handler) => handler(event));
    });
    ws.addEventListener('error', (event) => {
        handlers.error.forEach((handler) => handler(event));
    });
    return {
        on: ((event, handler) => {
            handlers[event].add(handler);
        }),
        send(data) {
            const payload = typeof data === 'string' ? data : JSON.stringify(data);
            ws.send(payload);
        },
        close(code, reason) {
            ws.close(code, reason);
        },
    };
}
//# sourceMappingURL=websocket.js.map