import { StructuredError } from './error';
export const WorkbenchConfigError = StructuredError('WorkbenchConfigError', 'The workbench configuration is invalid');
export const WorkbenchNotFoundError = StructuredError('WorkbenchNotFoundError', 'Workbench config not found - build process did not inline config');
/**
 * Encode workbench config to base64 for environment variable storage
 */
export function encodeWorkbenchConfig(config) {
    const json = JSON.stringify(config);
    // Use Node.js Buffer if available (build-time), otherwise browser btoa (shouldn't be called in browser)
    if (typeof Buffer !== 'undefined') {
        return Buffer.from(json).toString('base64');
    }
    else {
        return btoa(json);
    }
}
/**
 * Decode workbench config from base64 environment variable
 * Throws error if config is invalid
 */
export function decodeWorkbenchConfig(encoded) {
    try {
        let json;
        // Use appropriate decoding method based on environment
        if (typeof Buffer !== 'undefined') {
            // Node.js environment (build-time)
            json = Buffer.from(encoded, 'base64').toString('utf-8');
        }
        else if (typeof atob !== 'undefined') {
            // Browser environment (runtime)
            json = atob(encoded);
        }
        else {
            throw new Error('No base64 decoding method available');
        }
        const config = JSON.parse(json);
        return config;
    }
    catch (error) {
        throw new WorkbenchConfigError({
            cause: error,
        });
    }
}
/**
 * Get workbench config from build-time variable
 * Throws error if config is not available or invalid
 */
export function getWorkbenchConfig() {
    // This will be replaced at build time by Bun's define mechanism
    // @ts-expect-error - AGENTUITY_WORKBENCH_CONFIG_INLINE will be replaced at build time
    if (typeof AGENTUITY_WORKBENCH_CONFIG_INLINE === 'undefined') {
        throw new WorkbenchNotFoundError();
    }
    // @ts-expect-error - AGENTUITY_WORKBENCH_CONFIG_INLINE will be replaced at build time
    return decodeWorkbenchConfig(AGENTUITY_WORKBENCH_CONFIG_INLINE);
}
//# sourceMappingURL=workbench-config.js.map