/**
 * @module queue
 *
 * Queue service for publishing messages to Agentuity queues.
 *
 * This module provides a simplified interface for agents to publish messages
 * to queues. For full queue management (CRUD, consume, acknowledge), use
 * the `@agentuity/server` package.
 *
 * @example Publishing from an agent
 * ```typescript
 * // Inside an agent handler
 * const result = await ctx.queue.publish('order-queue', {
 *   orderId: 123,
 *   action: 'process',
 * });
 * console.log(`Published message ${result.id}`);
 * ```
 */
import { buildUrl, toServiceException, toPayload } from './_util';
import { StructuredError } from '../error';
// ============================================================================
// Errors
// ============================================================================
/**
 * Error thrown when a publish operation fails.
 *
 * This is a general error for publish failures that aren't specifically
 * validation or not-found errors.
 */
export const QueuePublishError = StructuredError('QueuePublishError');
/**
 * Error thrown when a queue is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await ctx.queue.publish('non-existent', 'payload');
 * } catch (error) {
 *   if (error instanceof QueueNotFoundError) {
 *     console.error('Queue does not exist');
 *   }
 * }
 * ```
 */
export const QueueNotFoundError = StructuredError('QueueNotFoundError');
/**
 * Error thrown when validation fails.
 *
 * Contains the field name and optionally the invalid value for debugging.
 */
export const QueueValidationError = StructuredError('QueueValidationError')();
// ============================================================================
// Internal Validation
// ============================================================================
const MAX_QUEUE_NAME_LENGTH = 256;
const MAX_PAYLOAD_SIZE = 1048576;
const MAX_PARTITION_KEY_LENGTH = 256;
const MAX_IDEMPOTENCY_KEY_LENGTH = 256;
const VALID_QUEUE_NAME_REGEX = /^[a-z_][a-z0-9_-]*$/;
/** @internal */
function validateQueueNameInternal(name) {
    if (!name || name.length === 0) {
        throw new QueueValidationError({
            message: 'Queue name cannot be empty',
            field: 'queueName',
            value: name,
        });
    }
    if (name.length > MAX_QUEUE_NAME_LENGTH) {
        throw new QueueValidationError({
            message: `Queue name must not exceed ${MAX_QUEUE_NAME_LENGTH} characters`,
            field: 'queueName',
            value: name,
        });
    }
    if (!VALID_QUEUE_NAME_REGEX.test(name)) {
        throw new QueueValidationError({
            message: 'Queue name must start with a letter or underscore and contain only lowercase letters, digits, underscores, and hyphens',
            field: 'queueName',
            value: name,
        });
    }
}
/** @internal */
function validatePayloadInternal(payload) {
    if (!payload || payload.length === 0) {
        throw new QueueValidationError({
            message: 'Payload cannot be empty',
            field: 'payload',
        });
    }
    if (payload.length > MAX_PAYLOAD_SIZE) {
        throw new QueueValidationError({
            message: `Payload size exceeds ${MAX_PAYLOAD_SIZE} byte limit (${payload.length} bytes)`,
            field: 'payload',
            value: payload.length,
        });
    }
}
// ============================================================================
// QueueStorageService Implementation
// ============================================================================
/**
 * HTTP-based implementation of the QueueService interface.
 *
 * This service communicates with the Agentuity Queue API to publish messages.
 * It is automatically configured and available via `ctx.queue` in agent handlers.
 *
 * @internal This class is instantiated by the runtime; use `ctx.queue` instead.
 */
export class QueueStorageService {
    #adapter;
    #baseUrl;
    /**
     * Creates a new QueueStorageService.
     *
     * @param baseUrl - The base URL of the Queue API
     * @param adapter - The fetch adapter for making HTTP requests
     */
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    /**
     * @inheritdoc
     */
    async publish(queueName, payload, params) {
        // Validate inputs before sending to API
        validateQueueNameInternal(queueName);
        const [body] = await toPayload(payload);
        const payloadStr = typeof payload === 'string' ? payload : body;
        validatePayloadInternal(payloadStr);
        // Validate optional params
        if (params?.partitionKey && params.partitionKey.length > MAX_PARTITION_KEY_LENGTH) {
            throw new QueueValidationError({
                message: `Partition key must not exceed ${MAX_PARTITION_KEY_LENGTH} characters`,
                field: 'partitionKey',
                value: params.partitionKey.length,
            });
        }
        if (params?.idempotencyKey && params.idempotencyKey.length > MAX_IDEMPOTENCY_KEY_LENGTH) {
            throw new QueueValidationError({
                message: `Idempotency key must not exceed ${MAX_IDEMPOTENCY_KEY_LENGTH} characters`,
                field: 'idempotencyKey',
                value: params.idempotencyKey.length,
            });
        }
        if (params?.ttl !== undefined && params.ttl < 0) {
            throw new QueueValidationError({
                message: 'TTL cannot be negative',
                field: 'ttl',
                value: params.ttl,
            });
        }
        const basePath = `/queue/messages/publish/2026-01-15/${encodeURIComponent(queueName)}`;
        const url = buildUrl(this.#baseUrl, params?.sync ? `${basePath}?sync=true` : basePath);
        const requestBody = {
            payload: typeof payload === 'string' ? payload : body,
        };
        if (params?.metadata) {
            requestBody.metadata = params.metadata;
        }
        if (params?.partitionKey) {
            requestBody.partition_key = params.partitionKey;
        }
        if (params?.idempotencyKey) {
            requestBody.idempotency_key = params.idempotencyKey;
        }
        if (params?.ttl !== undefined) {
            requestBody.ttl_seconds = params.ttl;
        }
        if (params?.projectId) {
            requestBody.project_id = params.projectId;
        }
        if (params?.agentId) {
            requestBody.agent_id = params.agentId;
        }
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            body: JSON.stringify(requestBody),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.queue.publish',
                attributes: {
                    queueName,
                },
            },
        });
        if (res.ok) {
            const data = res.data;
            return {
                id: data.message.id,
                offset: data.message.offset,
                publishedAt: data.message.published_at,
            };
        }
        if (res.response.status === 404) {
            throw new QueueNotFoundError({
                message: `Queue not found: ${queueName}`,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
}
//# sourceMappingURL=queue.js.map