import { buildUrl, toServiceException, toPayload } from './_util';
/**
 * Minimum TTL value in seconds (1 minute)
 */
export const KV_MIN_TTL_SECONDS = 60;
/**
 * Maximum TTL value in seconds (90 days)
 */
export const KV_MAX_TTL_SECONDS = 7776000;
/**
 * Default TTL value in seconds (7 days) - used when namespace is auto-created or no TTL specified
 */
export const KV_DEFAULT_TTL_SECONDS = 604800;
export class KeyValueStorageService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async get(name, key) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.get',
                attributes: {
                    name,
                    key,
                },
            },
        });
        if (res.ok) {
            const expiresAt = res.response.headers.get('x-expires-at') ?? undefined;
            return {
                data: res.data,
                contentType: res.response.headers.get('content-type') ?? 'application/octet-stream',
                exists: true,
                ...(expiresAt && { expiresAt }),
            };
        }
        if (res.response.status === 404) {
            return { exists: false };
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * set a value in the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to set the value of
     * @param value - the value to set in any of the supported data types
     * @param params - the KeyValueStorageSetParams
     *
     * @remarks
     * TTL behavior:
     * - If TTL is not specified (undefined), the key inherits the namespace's default TTL
     * - If TTL is null or 0, the key will not expire
     * - If TTL is a positive number, the key expires after that many seconds
     * - TTL values below 60 seconds are clamped to 60 seconds by the server
     * - TTL values above 7,776,000 seconds (90 days) are clamped to 90 days
     * - If the namespace doesn't exist, it is auto-created with a 7-day default TTL
     */
    async set(name, key, value, params) {
        // TTL handling: only include if explicitly provided
        // null or 0 = no expiration (send 0 to server), positive = TTL in seconds
        // undefined = not sent, server uses namespace default
        let ttlstr = '';
        if (params?.ttl !== undefined) {
            const ttlValue = params.ttl === null ? 0 : params.ttl;
            ttlstr = `/${ttlValue}`;
        }
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}${ttlstr}`);
        const [body, contentType] = await toPayload(value);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PUT',
            signal,
            body,
            contentType: params?.contentType || contentType,
            telemetry: {
                name: 'agentuity.keyvalue.set',
                attributes: {
                    name,
                    key,
                    ttl: ttlstr,
                },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('PUT', url, res.response);
    }
    async delete(name, key) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.delete',
                attributes: {
                    name,
                    key,
                },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async getStats(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/stats/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getStats',
                attributes: { name },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getAllStats(params) {
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined) {
            queryParams.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            queryParams.set('offset', String(params.offset));
        }
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/stats${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getAllStats',
                attributes: {},
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getNamespaces() {
        const url = buildUrl(this.#baseUrl, '/kv/2025-03-17/namespaces');
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getNamespaces',
                attributes: {},
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async search(name, keyword) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/search/${encodeURIComponent(name)}/${encodeURIComponent(keyword)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.search',
                attributes: { name, keyword },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getKeys(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/keys/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getKeys',
                attributes: { name },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteNamespace(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.deleteNamespace',
                attributes: { name },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async createNamespace(name, params) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const body = params?.defaultTTLSeconds !== undefined
            ? JSON.stringify({ default_ttl_seconds: params.defaultTTLSeconds })
            : undefined;
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            ...(body && { body, contentType: 'application/json' }),
            telemetry: {
                name: 'agentuity.keyvalue.createNamespace',
                attributes: { name },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('POST', url, res.response);
    }
}
//# sourceMappingURL=keyvalue.js.map