import type { Logger } from '@agentuity/core';
/**
 * Git information detected from the repository
 */
export interface GitInfo {
    branch?: string;
    repo?: string;
    provider?: string;
    tags?: string[];
    commit?: string;
    message?: string;
}
/**
 * Git options that can be provided via CLI flags to override auto-detected values
 */
export interface GitOptions {
    message?: string;
    commit?: string;
    branch?: string;
    repo?: string;
    provider?: string;
    commitUrl?: string;
}
/**
 * Extended git info that includes CI-related fields
 */
export interface GitInfoExtended extends GitInfo {
    commitUrl?: string;
    logsUrl?: string;
    trigger?: string;
    event?: string;
    pull_request?: {
        number: number;
        url?: string;
    };
}
/**
 * Detect git information from the repository at the given root directory.
 * Walks up parent directories to find .git directory (supports monorepos).
 *
 * @param rootDir - The root directory to start searching for .git
 * @param logger - Logger instance for trace output
 * @returns Git information or undefined if not in a git repository
 */
export declare function getGitInfo(rootDir: string, logger: Logger): Promise<GitInfo | undefined>;
/**
 * Merge auto-detected git info with CLI-provided overrides.
 * CLI options take precedence over auto-detected values.
 *
 * @param autoDetected - Git info auto-detected from the repository
 * @param overrides - CLI options that override auto-detected values
 * @returns Merged git info
 */
export declare function mergeGitInfo(autoDetected: GitInfo | undefined, overrides: GitOptions): GitInfoExtended;
/**
 * Build an array of tags from git info, including branch and short commit.
 * Used for deployment tagging.
 *
 * @param gitInfo - Git information
 * @returns Array of tags
 */
export declare function buildGitTags(gitInfo: GitInfo | undefined): string[];
//# sourceMappingURL=git.d.ts.map